<?php
    /**
     *  Module Name: DM Post Type Manager
     *  Version:     v1.0
     *  Description: An admin interface for managing custom post types.
     *  Author:      Distill Mill LLC
     *  Author URI:  http://distillmill.com
     */

    class DM_PostTypeManager{
        private static $instance;
        public $postType = 'dm_post_types';

        public static function getInstance( ){
            if( !self::$instance )
                new self;

            return self::$instance;
        }

        public static function call( $method ){
            return array( self::getInstance( ), $method );
        }

        public function __construct( ){
            if( self::$instance )
                return;
            self::$instance = $this;

            register_post_type(
                $this->postType,
                array(
                    'label'                 => 'DM Post Types',
                    'public'                => false,
                    'exclude_from_search'   => true,
                    'show_ui'               => true,
                    'rewrite'               => false,
                    'supports'              => array( 'title' )
                )
            );

            add_action( 'init', self::call( 'loadPostTypes' ) );

            // Add custom message strings
            add_filter( 'post_updated_messages',        self::call( 'addUpdatedMessages' ) );
            add_filter( 'bulk_post_updated_messages',   self::call( 'addBulkUpdatedMessages'), 10, 2 );
            add_filter( 'the_posts',                    self::call( 'setupPostTypeListPage' ), 10, 2 );
            add_action( 'admin_enqueue_scripts', self::call( 'enqueueScripts' ) );
        }

        public function enqueueScripts( $hook ){
            $screen = get_current_screen( );
            if( $screen->post_type == $this->postType ){
                wp_enqueue_style( 'dm_admin_css' );
                wp_enqueue_script( 'dm_admin_js' );
            }
        }

        public function loadPostTypes( ){
            $types = $this->getCustomTypes( );
            foreach( $types as $type ){
                $args = $type->args;
                $slug = $type->getSlug( );

                if( array_key_exists( 'menu_position', $args ) )
                    $args[ 'menu_position' ] = intval( $args[ 'menu_position' ] );

                register_post_type( $slug, $args );
            }

            if( get_option( 'dmptm_flush_rewrite' ) == true ){
                flush_rewrite_rules( );
                update_option( 'dmptm_flush_rewrite', false );
            }
        }

        public function addUpdatedMessages( $messages ){
            global $post;
            global $post_ID;
            $types = $this->getCustomTypes( );
            foreach( $types as $type ){
                $rawStrings = $type->getStrings( 'updated' );
                if( count( $rawStrings ) > 0 ){
                    $strings = array( );
                    foreach( $rawStrings as $index=>$string ){
                        if( isset( $_GET[ 'revision' ] ) )
                            $string = str_replace( '%revision%', wp_post_revision_title( (int) $_GET[ 'revision' ], false ), $string );
                        if( $post )
                            $string = str_replace( '%publish%', date_i18n( __( 'M j, Y @ G:i' ), strtotime( $post->post_date ) ), $string );
                        $strings[ $index ] = str_replace( '%url%', esc_url( get_permalink( $post_ID ) ), $string );
                    }
                    $messages[ $type->getSlug( ) ] = array_replace( $messages[ 'post' ], $strings );
                }
            }

            return $messages;
        }

        public function addBulkUpdatedMessages( $messages, $counts ){
            $types = $this->getCustomTypes( );
            foreach( $types as $type ){
                $rawStrings = $type->getStrings( 'bulk_updated' );
                if( count( $rawStrings ) > 0 ){
                    $strings = array( );
                    foreach( $rawStrings as $slug=>$string ){
                        $slug = explode( '-', $slug );
                        $plural = array_pop( $slug ) == 'plural';
                        $slug = implode( '-', $slug );
                        if( ( $plural && $counts[ $slug ] > 1 ) ||
                            ( !$plural && $counts[ $slug ] == 1 ) )
                            $strings[ $slug ] = $string;
                    }
                    $messages[ $type->getSlug( ) ] = array_replace( $messages[ 'post' ], $strings );
                }
            }

            return $messages;
        }

        public function getCustomTypes( ){
            $rawTypes = get_posts( array(
                'post_type'         => 'dm_post_types',
                'post_status'       => 'publish',
                'posts_per_page'    => -1
            ) );

            $types = array( );
            foreach( $rawTypes as $type ){
                $types[ ] = new DM_PostType( $type );
            }

            return $types;
        }

        // This method 
        public function setupPostTypeListPage( $posts, $query ){
            if( $posts && $query->is_main_query( ) && $query->is_page( ) ){
                $types = $this->getCustomTypes( );
                foreach( $types as $type ){
                    $showOn = keyCheck( $type->args, 'show_on', 'all_pages' );
                    if( keyCheck( $type->args, 'list_on' ) === $query->queried_object_id ){
                        $postsPerPage = (int) keyCheck( $type->args, 'posts_per_page' );
                        $posts = new WP_Query( array(
                            'post_type'      => $type->getSlug( ),
                            'post_status'    => 'publish',
                            'posts_per_page' => ( $postsPerPage > 0 ? $postsPerPage : get_option( 'posts_per_page' ) ),
                            'paged'          => $query->query_vars[ 'paged' ]
                        ) );
                        if( ( $showOn == 'first_page' && $query->query_vars[ 'paged' ] > 1 )
                           || $showOn == 'no_pages' ){
                            $query->found_posts = $posts->found_posts;
                            $query->posts = $posts->posts;
                        } else{
                            $query->found_posts += $posts->found_posts;
                            $query->posts = array_merge( $query->posts, $posts->posts );
                        }
                        $query->max_num_pages = $posts->max_num_pages;
                        $posts = $query->posts;
                        break;
                    }
                }
            }
            return $posts;
        }
    }

    class DM_PostType{
        private $post;
        private $slug;

        public $args;
        public $strings;

        public function __construct( $post ){
            $this->args     = get_post_meta( $post->ID, 'dm_post_types_args', true );
            $this->post     = $post;
            $this->slug     = get_post_meta( $post->ID, 'dm_post_types_slug', true );
            $this->strings  = get_post_meta( $post->ID, 'dm_post_types_strings', true );
        }

        public function getSlug( ){
            return $this->slug;
        }

        public function getLabel( $name ){
            $labels = $this->getOption( 'labels' );
            if( $labels && array_key_exists( $name, $labels ) )
                return $labels[ $name ];

            return '';
        }

        public function getStrings( $variety ){
            if( is_array( $this->strings ) ){
                if( array_key_exists( $variety, $this->strings ) )
                    return $this->strings[ $variety ];
            }

            return array( );
        }

        public function getString( $variety, $name ){
            $strings = $this->getStrings( $variety );
            if( count( $strings ) > 0 && array_key_exists( $name, $strings ) )
                return $strings[ $name ];

            return '';
        }

        public function getOption( $name ){
            if( $this->args && array_key_exists( $name, $this->args ) )
                return $this->args[ $name ];

            return null;
        }

        public function getPost( ){
            return $this->post;
        }

        public function setLabels( $labels ){
            $this->setOption( 'labels', $labels );
        }

        public function setOption( $name, $value ){
            if( !is_array( $this->args ) )
                $this->args = array( );

            if( !is_null( $value ) )
                $this->args[ $name ] = $value;
            elseif( array_key_exists( $name, $this->args ) )
                unset( $this->args[ $name ] );
        }

        public function setSlug( $slug ){
            $this->slug = $slug;
        }

        public function save( ){
            update_post_meta( $this->post->ID, 'dm_post_types_args',    $this->args );
            update_post_meta( $this->post->ID, 'dm_post_types_slug',    $this->slug );
            update_post_meta( $this->post->ID, 'dm_post_types_strings', $this->strings );
        }
    }

    class DM_PostTypeAdmin{
        private static $instance;
        private $labels = array( );
        private $options = array( );

        public static function getInstance( ){
            if( !self::$instance )
                new self;

            return self::$instance;
        }

        public static function call( $method ){
            return array( self::getInstance( ), $method );
        }

        public function __construct( ){
            if( self::$instance )
                return;
            self::$instance = $this;

            add_action( 'add_meta_boxes',                           self::call( 'addMetaBoxes' ) );
            add_action( 'save_post',                                self::call( 'savePostType' ) );
            add_action( 'manage_dm_post_types_posts_custom_column', self::call( 'postTypeColumns' ), 10, 2 );
            add_filter( 'manage_dm_post_types_posts_columns',       self::call( 'postTypeColumnHeaders' ) );
            add_filter( 'post_row_actions',                         self::call( 'removeQuickEdit' ), 10, 2 );
            add_filter( 'bulk_actions-edit-dm_post_types',          self::call( 'removeEditBulkOption' ) );
        }

        public function addMetaBoxes( ){
            DM()->admin->addMetaBox( 'dm_post_type_manager_options',  'Options',          self::call( 'showPostTypeOptions' ),  'dm-meta-box no-pad', 'dm_post_types', 'normal',   'high' );
            DM()->admin->addMetaBox( 'dm_post_type_manager_labels',   'Labels',           self::call( 'showPostTypeLabels' ),   'dm-meta-box no-pad', 'dm_post_types', 'normal',   'high' );
            DM()->admin->addMetaBox( 'dm_post_type_manager_strings',  'Strings',          self::call( 'showPostTypeStrings' ),  'dm-meta-box no-pad', 'dm_post_types', 'normal',   'high' );
            DM()->admin->addMetaBox( 'dm_post_type_manager_advanced', 'Advanced Options', self::call( 'showPostTypeAdvanced' ), 'dm-meta-box no-pad', 'dm_post_types', 'advanced', 'high' );
            
            DM()->admin->addMetaBox( 'dm_post_type_publish', 'Publish', self::call( 'showPostTypePublish' ), 'dm-meta-box no-pad', 'dm_post_types', 'side', 'high' );

            remove_meta_box( 'submitdiv', 'dm_post_types', 'side' );  // Removes the default publish meta box
        }

        public function removeQuickEdit( $actions, $post ){
            if( $post->post_type == 'dm_post_types' )
                unset( $actions[ 'inline hide-if-no-js' ] );

            return $actions;
        }

        public function removeEditBulkOption( $actions ){
            unset( $actions[ 'edit' ] );
            return $actions;
        }

        public function postTypeColumns( $columnName, $postID ){
            $pt = new DM_PostType( get_post( $postID ) );
            switch( $columnName ){
                case 'dm-post-type-slug':
                    echo $pt->getSlug( );
                    break;
                case 'dm-post-type-public':
                    echo ( $pt->getOption( 'public' ) ? 'Yes' : 'No' );
                    break;
                case 'dm-post-type-admin':
                    echo ( $pt->getOption( 'show_ui' ) ? 'Yes' : 'No' );
                    break;
            }
        }

        public function postTypeColumnHeaders( $columns ){
            $columns[ 'dm-post-type-slug' ] = 'Slug';
            $columns[ 'dm-post-type-public' ] = 'Public';
            $columns[ 'dm-post-type-admin' ] = 'Show UI';

            unset( $columns[ 'date' ] );  // Don't want date showing up

            return $columns;
        }

        private function constructForms( ){
            $this->labels = array(  'name'                 => array( 'label' => 'Name' ),
                                    'singular_name'        => array( 'label' => 'Singular Name' ),
                                    'menu_name'            => array( 'label' => 'Menu Name' ),
                                    'name_admin_bar'       => array( 'label' => 'Name Admin Bar' ),
                                    'all_items'            => array( 'label' => 'All Items' ),
                                    'add_new'              => array( 'label' => 'Add New' ),
                                    'add_new_item'         => array( 'label' => 'Add New Item' ),
                                    'edit_item'            => array( 'label' => 'Edit Item' ),
                                    'new_item'             => array( 'label' => 'New Item' ),
                                    'view_item'            => array( 'label' => 'View Item' ),
                                    'search_items'         => array( 'label' => 'Search Items' ),
                                    'not_found'            => array( 'label' => 'Not Found' ),
                                    'not_found_in_trash'   => array( 'label' => 'Not Found In Trash' ),
                                    'parent_item_colon'    => array( 'label' => 'Parent Item Colon' ) );

            $this->strings = array(
                'updated'       => array(   'post_updated_link'     => 'Post updated. <u>View post</u>',
                                            'field_updated'         => 'Custom field updated.',
                                            'field_deleted'         => 'Custom field deleted.',
                                            'post_updated'          => 'Post updated.',
                                            'restore_revision'      => 'Post restored to revision from %revision%',
                                            'post_published'        => 'Post published. <u>View post</u>',
                                            'post_saved'            => 'Post saved.',
                                            'post_submitted'        => 'Post submitted. <u>Preview post</u>',
                                            'post_scheduled'        => 'Post scheduled for: <strong>%publish%</strong>. <u>Preview post</u>',
                                            'draft_updated'         => 'Post draft updated. <u>Preview post</u>' ),
                'bulk_updated'  => array(   'updated'   => array( 'singular' => '%s post updated.',
                                                                  'plural'   => '%s posts updated.' ),
                                            'locked'    => array( 'singular' => '%s post not updated, somebody is editing it.',
                                                                  'plural'   => '%s posts not updated, somebody is editing them.' ),
                                            'deleted'   => array( 'singular' => '%s post permanently deleted.',
                                                                  'plural'   => '%s posts permanently deleted.' ),
                                            'trashed'   => array( 'singular' => '%s post moved to the Trash.',
                                                                  'plural'   => '%s posts moved to the Trash.' ),
                                            'untrashed' => array( 'singular' => '%s post restored from the Trash.',
                                                                  'plural'   => '%s posts restored from the Trash.' ) ) );

            $this->options = array( 'public'                => array(   'label'     => 'Public',
                                                                        'type'      => 'radio',
                                                                        'display'   => 'inline',
                                                                        'default'   => null,
                                                                        'options'   => array(   'true'      => array(   'label' => 'True',
                                                                                                                        'save'  => true ),
                                                                                                'false'     => array(   'label' => 'False',
                                                                                                                        'save'  => false ),
                                                                                                'default'   => array(   'label' => 'Default',
                                                                                                                        'save'  => null ) ) ),
                                    'exclude_from_search'   => array(   'label'     => 'Exclude From Search',
                                                                        'type'      => 'radio',
                                                                        'display'   => 'inline',
                                                                        'default'   => null,
                                                                        'options'   => array(   'true'      => array(   'label' => 'True',
                                                                                                                        'save'  => true ),
                                                                                                'false'     => array(   'label' => 'False',
                                                                                                                        'save'  => false ),
                                                                                                'default'   => array(   'label' => 'Default',
                                                                                                                        'save'  => null ) ) ),
                                    'publicly_queryable'    => array(   'label'     => 'Publicly Queryable',
                                                                        'type'      => 'radio',
                                                                        'display'   => 'inline',
                                                                        'default'   => null,
                                                                        'options'   => array(   'true'      => array(   'label' => 'True',
                                                                                                                        'save'  => true ),
                                                                                                'false'     => array(   'label' => 'False',
                                                                                                                        'save'  => false ),
                                                                                                'default'   => array(   'label' => 'Default',
                                                                                                                        'save'  => null ) ) ),
                                    'show_ui'               => array(   'label'     => 'Show UI',
                                                                        'type'      => 'radio',
                                                                        'display'   => 'inline',
                                                                        'default'   => null,
                                                                        'options'   => array(   'true'      => array(   'label' => 'True',
                                                                                                                        'save'  => true ),
                                                                                                'false'     => array(   'label' => 'False',
                                                                                                                        'save'  => false ),
                                                                                                'default'   => array(   'label' => 'Default',
                                                                                                                        'save'  => null ) ) ),
                                    'show_in_nav_menus'     => array(   'label'     => 'Show In Navigation',
                                                                        'type'      => 'radio',
                                                                        'display'   => 'inline',
                                                                        'default'   => null,
                                                                        'options'   => array(   'true'      => array(   'label' => 'True',
                                                                                                                        'save'  => true ),
                                                                                                'false'     => array(   'label' => 'False',
                                                                                                                        'save'  => false ),
                                                                                                'default'   => array(   'label' => 'Default',
                                                                                                                        'save'  => null ) ) ),
                                    'show_in_admin_bar'     => array(   'label'     => 'Show In Admin Bar',
                                                                        'type'      => 'radio',
                                                                        'display'   => 'inline',
                                                                        'default'   => null,
                                                                        'options'   => array(   'true'      => array(   'label' => 'True',
                                                                                                                        'save'  => true ),
                                                                                                'false'     => array(   'label' => 'False',
                                                                                                                        'save'  => false ),
                                                                                                'default'   => array(   'label' => 'Default',
                                                                                                                        'save'  => null ) ) ),
                                    'hierarchical'          => array(   'label'     => 'Hierarchical',
                                                                        'type'      => 'radio',
                                                                        'display'   => 'inline',
                                                                        'default'   => null,
                                                                        'options'   => array(   'true'      => array(   'label' => 'True',
                                                                                                                        'save'  => true ),
                                                                                                'false'     => array(   'label' => 'False',
                                                                                                                        'save'  => false ),
                                                                                                'default'   => array(   'label' => 'Default',
                                                                                                                        'save'  => null ) ) ),
                                    'has_archive'           => array(   'label'     => 'Has Archive',
                                                                        'type'      => 'radio',
                                                                        'display'   => 'inline',
                                                                        'default'   => null,
                                                                        'options'   => array(   'true'      => array(   'label' => 'True',
                                                                                                                        'save'  => true ),
                                                                                                'false'     => array(   'label' => 'False',
                                                                                                                        'save'  => false ),
                                                                                                'default'   => array(   'label' => 'Default',
                                                                                                                        'save'  => null ) ) ),
                                    'can_export'            => array(   'label'     => 'Can Export',
                                                                        'type'      => 'radio',
                                                                        'display'   => 'inline',
                                                                        'default'   => null,
                                                                        'options'   => array(   'true'      => array(   'label' => 'True',
                                                                                                                        'save'  => true ),
                                                                                                'false'     => array(   'label' => 'False',
                                                                                                                        'save'  => false ),
                                                                                                'default'   => array(   'label' => 'Default',
                                                                                                                        'save'  => null ) ) ),
                                    
                                    'show_in_menus'         => array(   'label'     => 'Show In Admin Menu',
                                                                        'type'      => 'editable_select',
                                                                        'values'    => array(   'true'  => 'True',
                                                                                                'false' => 'False' ) ),
                                    'menu_position'         => array(   'label'     => 'Menu Position',
                                                                        'type'      => 'editable_select',
                                                                        'values'    => array(   5     => '5 - below Posts',
                                                                                                10    => '10 - below Media',
                                                                                                15    => '15 - below Links',
                                                                                                20    => '20 - below Pages',
                                                                                                25    => '25 - below comments',
                                                                                                60    => '60 - below first seperator',
                                                                                                65    => '65 - below Plugins',
                                                                                                70    => '70 - below Users',
                                                                                                75    => '75 - below Tools',
                                                                                                80    => '80 - below Settings',
                                                                                                100   => '100 - below second seperator' ) ),
                                    'menu_icon'             => array(   'label'     => 'Menu Icon',
                                                                        'type'      => 'callback',
                                                                        'render'    => self::call( 'renderMenuIcons' ) ),
                                    'supports'              => array(   'label'     => 'Supports',
                                                                        'type'      => 'multi_select',
                                                                        'desc'      => 'CMD/CTRL click to select multiple.',
                                                                        'values'    => array(   'title'             => 'Title',
                                                                                                'editor'            => 'Editor (content)',
                                                                                                'author'            => 'Author',
                                                                                                'thumbnail'         => 'Thumbnail',
                                                                                                'excerpt'           => 'Excerpt',
                                                                                                'trackbacks'        => 'Trackbacks',
                                                                                                'custom-fields'     => 'Custom Fields',
                                                                                                'comments'          => 'Comments',
                                                                                                'revisions'         => 'Revisions',
                                                                                                'page-attributes'   => 'Page Attributes',
                                                                                                'post-formats'      => 'Post Formats' ) ),
                                    'taxonomies'            => array(   'label'     => 'Taxonomies',
                                                                        'type'      => 'multi_select',
                                                                        'desc'      => 'CMD/CTRL click to select multiple.',
                                                                        'values'    => $this->getTaxonomiesList( ) ) );
            $this->advancedOptions = array(
                'map_meta_cap'  => array(
                    'label'     => 'Map Meta Capability',
                    'type'      => 'radio',
                    'display'   => 'inline',
                    'default'   => null,
                    'options'   => array(
                        'true'      => array(
                            'label' => 'True',
                            'save'  => true
                        ),
                        'false'     => array(
                            'label' => 'False',
                            'save'  => false
                        ),
                        'default'   => array(
                            'label' => 'Default',
                            'save'  => null
                        )
                    )
                ),
                'capability_type' => array(
                    'label'     => 'Capability Type',
                    'type'      => 'callback',
                    'render'    => self::call( 'renderCapabilityType' ),
                    'desc'      => 'These are used by WordPress to generate any capabilities not set below.'
                ),
                'capabilities' => array(
                    'label'     => 'Capabilities',
                    'type'      => 'repeat',
                    'desc'      => 'Post capabilities are actions that can be performed if the user\'s role has the user capability.',
                    'addText'   => 'Add Capability',
                    'noneText'  => 'No Capabilities Set',
                    'editKey'   => true,
                    'fields'    => array(
                        'capabilities'      => array(
                            'label' => 'Capability',
                            'type'  => 'keyvalue'
                        )
                    )
                ),
                'permalink_epmask' => array(
                    'label'     => 'Permalink Endpoint Mask',
                    'type'      => 'text'
                ),
                'query_var' => array(
                    'label'     => 'Query Var',
                    'type'      => 'editable_select',
                    'values'    => array( 'false' => 'Default' )
                ),
                'register_meta_box_cb' => array(
                    'label'     => 'Register Meta Box Callback',
                    'type'      => 'text'
                ),
                'rewrite' => array(
                    'label'     => 'Rewrite',
                    'type'      => 'advanced',
                    'fields'    => array(
                        'slug' => array(
                            'label' => 'Slug',
                            'type'  => 'text'
                        ),
                        'ep_mask' => array(
                            'label' => 'Endpoint Mask',
                            'type'  => 'text'
                        ),
                        'with_front' => array(
                            'label' => 'With Front',
                            'type'  => 'checkbox'
                        ),
                        'feeds' => array(
                            'label' => 'Feeds',
                            'type'  => 'checkbox'
                        )
                    )
                )
            );
        }

        public function savePostType( $postID ){
            if( !isset( $_POST[ 'dmptm_nonce' ] ) || !wp_verify_nonce( $_POST[ 'dmptm_nonce' ], 'dmptm_editing_pt' ) ) {
                return;
            }

            if( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
                return;
            }

            if( defined( 'DOING_AJAX' ) && DOING_AJAX ) {
                return;
            }

            if( defined( 'DOING_CRON' ) && DOING_CRON ) {
                return;
            }

            if( wp_is_post_revision( $postID ) ) {
                return;
            }

            if( !current_user_can( 'edit_post', $postID ) ) {
                return;
            }

            $this->constructForms( );
            $post = get_post( $postID );
            $pt = new DM_PostType( $post );
            $labels = array( );
            $strings = array(
                'updated'       => array( ),
                'bulk_updated'  => array( )
            );
            if( $post->post_type == 'dm_post_types' ){
                foreach( $this->labels as $name => $label ){
                    $id = 'dmptm_label_' . $name;
                    if( $_POST[ $id ] )
                        $labels[ $name ] = $_POST[ $id ];
                }
                $index = 1;
                foreach( $this->strings[ 'updated' ] as $name => $label ){
                    $id = 'dmptm_string_' . $name;
                    if( $_POST[ $id ] )
                        $strings[ 'updated' ][ $index ] = $_POST[ $id ];
                    $index++;
                }
                foreach( $this->strings[ 'bulk_updated' ] as $name => $label ){
                    $id = 'dmptm_string_bulk_' . $name . '-singular';
                    if( $_POST[ $id ] )
                        $strings[ 'bulk_updated' ][ $name . '-singular' ] = $_POST[ $id ];

                    $id = 'dmptm_string_bulk_' . $name . '-plural';
                    if( $_POST[ $id ] )
                        $strings[ 'bulk_updated' ][ $name . '-plural' ] = $_POST[ $id ];
                }
                foreach( $this->options as $name => $field ){
                    $id = 'dmptm_' . $name;
                    if( isset( $_POST[ $id ] ) && $_POST[ $id ] ){
                        $value = $_POST[ $id ];
                        switch( $field[ 'type' ] ){
                            case 'radio':
                                $value = $field[ 'options' ][ $value ][ 'save' ];
                                break;
                            case 'checkbox':
                                $value = ( $value == 'true' );
                                break;
                            case 'repeat':
                                if( isset( $field[ 'editKey' ] ) && $field[ 'editKey' ] ){
                                    $val = array( );
                                    for( $i=0; $i<count( $value ) / 2; $i++ ){
                                        $val[ $value[ $i*2 ][ 'key' ] ] = $value[ $i*2+1 ][ 'value' ];
                                    }
                                    $value = $val;
                                    break;
                                }
                                // else fall through
                            case 'multi_select':
                                $value = ( is_array( $value ) ? $value : null );
                                break;
                            case 'editable_select':
                                if( strtolower( $value ) == 'false' )
                                        $value = false;
                                elseif( strtolower( $value ) == 'true' )
                                        $value = true;
                                break;
                        }

                    } else{
                        $value = '';
                        switch( $field[ 'type' ] ){
                            case 'radio':
                                $value = $field[ 'default' ];
                                break;
                            case 'checkbox':
                                $value = false;
                                break;
                            case 'repeat':              // Fall through
                            case 'multi_select':
                                $value = null;
                                break;
                            case 'editable_select':
                                if( strtolower( $value ) == 'false' )
                                        $value = false;
                                elseif( strtolower( $value ) == 'true' )
                                        $value = true;
                                break;
                        }
                    }

                    $pt->setOption( $name, $value );
                }
                foreach( $this->advancedOptions as $name => $field ){
                    $id = 'dmptm_' . $name;
                    if( isset( $_POST[ $id ] ) && $_POST[ $id ] ){
                        $value = $_POST[ $id ];
                        switch( $field[ 'type' ] ){
                            case 'radio':
                                $value = $field[ 'options' ][ $value ][ 'save' ];
                                break;
                            case 'checkbox':
                                $value = ( $value == 'true' );
                                break;
                            case 'repeat':
                                if( isset( $field[ 'editKey' ] ) && $field[ 'editKey' ] ){
                                    $val = array( );
                                    for( $i=0; $i<count( $value ) / 2; $i++ ){
                                        $val[ $value[ $i*2 ][ 'key' ] ] = $value[ $i*2+1 ][ 'value' ];
                                    }
                                    $value = $val;
                                    break;
                                }
                                // else fall through
                            case 'multi_select':
                                $value = ( is_array( $value ) ? $value : null );
                                break;
                            case 'editable_select':
                                if( strtolower( $value ) == 'false' )
                                        $value = false;
                                elseif( strtolower( $value ) == 'true' )
                                        $value = true;
                                break;
                        }

                    } else{
                        $value = '';
                        switch( $field[ 'type' ] ){
                            case 'radio':
                                $value = $field[ 'default' ];
                                break;
                            case 'checkbox':
                                $value = false;
                                break;
                            case 'repeat':              // Fall through
                            case 'multi_select':
                                $value = null;
                                break;
                            case 'editable_select':
                                if( strtolower( $value ) == 'false' )
                                        $value = false;
                                elseif( strtolower( $value ) == 'true' )
                                        $value = true;
                                break;
                        }
                    }

                    $pt->setOption( $name, $value );
                }
                $rewrite = array( );
                $flush = false;
                foreach( $this->advancedOptions[ 'rewrite' ][ 'fields' ] as $name => $field ){
                    $id = 'dmptm_rewrite_' . $name;
                    if( isset( $_POST[ $id ] ) && $_POST[ $id ] ){
                        $value = $_POST[ $id ];
                        switch( $field[ 'type' ] ){
                            case 'checkbox':
                                $value = ( $value == 'true' );
                                break;
                        }

                    } else{
                        $value = '';
                        switch( $field[ 'type' ] ){
                            case 'checkbox':
                                $value = false;
                                break;
                        }
                    }

                    $flush = $flush || (bool) $value;
                    $rewrite[ $name ] = $value;
                }

                $pt->setOption( 'rewrite', $rewrite );
                update_option( 'dmptm_flush_rewrite', true );  // Needs to be done each time to ensure that post type rewrites work
                if( keyCheck( $_POST, 'dmptm_slug' ) )
                    $pt->setSlug( $_POST[ 'dmptm_slug' ] );
                if( isset( $_POST[ 'dmptm_description' ] ) )
                    $pt->setOption( 'description', $_POST[ 'dmptm_description' ] );
                if( isset( $_POST[ 'dmptm_list_on' ] ) )
                    $pt->setOption( 'list_on', (int) $_POST[ 'dmptm_list_on' ] );
                if( isset( $_POST[ 'dmptm_show_on' ] ) )
                    $pt->setOption( 'show_on', $_POST[ 'dmptm_show_on' ] );
                if( isset( $_POST[ 'dmptm_posts_per_page' ] ) )
                    $pt->setOption( 'posts_per_page', $_POST[ 'dmptm_posts_per_page' ] );
            }
            $pt->setLabels( $labels );
            $pt->strings = $strings;
            $pt->setOption( 'label', $post->post_title );

            $pt->save( );
        }

        public function showPostTypeLabels( $post ){
            $this->constructForms( );
            $pt = new DM_PostType( $post );
            wp_nonce_field( 'dmptm_editing_pt', 'dmptm_nonce' );
?>
            <table class="dm-settings-table">
                <tbody>
<?php
foreach( $this->labels as $slug => $label ){
    $id = 'dmptm_label_' . $slug;
    echo '<tr>';
    echo '<td class="left">';
    echo '<label for="' . $id . '">' . $label[ 'label' ] . '</label>';
    echo '</td>';
    echo '<td class="right">';
    echo '<input type="text" id="' . $id . '" name="' . $id . '"' . ( $pt->getLabel( $slug ) ? ' value="' . $pt->getLabel( $slug ) . '"' : '' ) . ' />';
    echo '</td>';
    echo '</tr>';
}
?>
                    <tr>
                        <td class="left">
                            <label for="dmptm_description">Description</label>
                        </td>
                        <td class="right">
                            <textarea id="dmptm_description" name="dmptm_description"><?php echo ( $pt->getOption( 'description' ) ? $pt->getOption( 'description' ) : '' ); ?></textarea>
                        </td>
                    </tr>
                </tbody>
            </table>
<?php
        }

        public function showPostTypeStrings( $post ){
            $this->constructForms( );
            $pt = new DM_PostType( $post );
            wp_nonce_field( 'dmptm_editing_pt', 'dmptm_nonce' );
?>
            <table class="dm-settings-table">
                <tbody>
                    <tr>
                        <td class="left heading">
                            <h3>Post Type Update</h3>
                        </td>
                        <td class="right heading">
                            <p>You can enter HTML and use %url% to get the permalink for the post
                            being updated. You can also use any %naming% listed below.</p>
                        </td>
                    </tr>
<?php
$index = 1;  // WordPress starts the update strings array at 1
foreach( $this->strings[ 'updated' ] as $slug => $label ){
    $id = 'dmptm_string_' . $slug;
    echo '<tr>';
    echo '<td class="left">';
    echo '<label for="' . $id . '">' . $label . '</label>';
    echo '</td>';
    echo '<td class="right">';
    echo '<input type="text" id="' . $id . '" name="' . $id . '"' . ( $pt->getString( 'updated', $index ) ? ' value="' . esc_html( $pt->getString( 'updated', $index ) ) . '"' : '' ) . ' />';
    echo '</td>';
    echo '</tr>';
    $index++;
}
?>
                    <tr>
                        <td class="left heading">
                            <h3>Bulk Post Type Update</h3>
                        </td>
                        <td class="right heading">
                            <p>You can enter HTML and each string <strong>must</strong> have a %s to display the number of posts effected.</p>
                        </td>
                    </tr>
<?php
// This one uses the slug rather than an index...
foreach( $this->strings[ 'bulk_updated' ] as $slug => $label ){
    $id = 'dmptm_string_bulk_' . $slug . '-singular';
    echo '<tr>';
    echo '<td class="left">';
    echo '<label for="' . $id . '">' . $label[ 'singular' ] . '</label>';
    echo '</td>';
    echo '<td class="right">';
    echo '<input type="text" id="' . $id . '" name="' . $id . '"' . ( $pt->getString( 'bulk_updated', $slug . '-singular' ) ? ' value="' . esc_html( $pt->getString( 'bulk_updated', $slug . '-singular' ) ) . '"' : '' ) . ' />';
    echo '</td>';
    echo '</tr>';

    $id = 'dmptm_string_bulk_' . $slug . '-plural';
    echo '<tr>';
    echo '<td class="left">';
    echo '<label for="' . $id . '">' . $label[ 'plural' ] . '</label>';
    echo '</td>';
    echo '<td class="right">';
    echo '<input type="text" id="' . $id . '" name="' . $id . '"' . ( $pt->getString( 'bulk_updated', $slug . '-plural' ) ? ' value="' . esc_html( $pt->getString( 'bulk_updated', $slug . '-plural' ) ) . '"' : '' ) . ' />';
    echo '</td>';
    echo '</tr>';
}
?>
                </tbody>
            </table>
<?php
        }

        public function showPostTypeOptions( $post ){
            $this->constructForms( );
            $pt = new DM_PostType( $post );
            $pages = get_posts( array(
                'post_type'   => 'page',
                'post_status' => 'publish'
            ) );
?>
            <table class="dm-settings-table">
                <tbody>
                    <tr>
                        <td class="left">
                            <label for="dmptm_slug">Post Type Slug</label>
                        </td>
                        <td class="right">
                            <input type="text" id="dmptm_slug" name="dmptm_slug"<?php echo ( $pt->getSlug( ) ? ' value="' . $pt->getSlug( ) .'"' : '' ); ?> />
                        </td>
                    </tr>
                    <tr>
                        <td class="left">
                            <label for="dmptm_list_on">List On</label>
                            <p class="description">This page will be updated to
                            list posts in this post type.</p>
                        </td>
                        <td class="right">
                            <select id="dmptm_list_on" name="dmptm_list_on">
                                <option value="0">None</option>
                                <?php
                                    foreach( $pages as $page ){
                                        echo '<option value="' . $page->ID . '"';
                                        if( keyCheck( $pt->args, 'list_on' ) === $page->ID )
                                            echo ' selected';
                                        echo '>' . $page->post_title . '</option>';
                                    }
                                ?>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <td class="left">
                            <label for="dmptm_list_on">Show Page Content On</label>
                            <p class="description">When to show the page's content. This is most apparent when paginating.</p>
                        </td>
                        <td class="right">
                            <select id="dmptm_show_on" name="dmptm_show_on">
                                <option value="all_pages"<?php if( keyCheck( $pt->args, 'show_on' ) == 'all_pages' ) echo ' selected'; ?>>All Pages</option>
                                <option value="first_page"<?php if( keyCheck( $pt->args, 'show_on' ) == 'first_page' ) echo ' selected'; ?>>First Page</option>
                                <option value="no_pages"<?php if( keyCheck( $pt->args, 'show_on' ) == 'no_pages' ) echo ' selected'; ?>>Never</option>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <td class="left">
                            <label for="dmptm_list_on">Posts Per Page</label>
                            <p class="description">How many posts to show per page. Leave blank for the WordPress setting.</p>
                        </td>
                        <td class="right">
                            <input id="dmptm_posts_per_page" name="dmptm_posts_per_page" value="<?php echo keyCheck( $pt->args, 'posts_per_page' ); ?>" />
                        </td>
                    </tr>
                    <?php $this->buildForm( $this->options, $pt->args ); ?>
                </tbody>
            </table>
<?php
        }

        public function showPostTypeAdvanced( $post ){
            $this->constructForms( );
            $pt = new DM_PostType( $post );
            $pages = get_posts( array(
                'post_type'   => 'page',
                'post_status' => 'publish'
            ) );
?>
            <table class="dm-settings-table">
                <tbody>
                    <?php $this->buildForm( $this->advancedOptions, $pt->args ); ?>
                </tbody>
            </table>
<?php
        }

        public function showPostTypePublish( $post ){
            global $action;

            $post_type = $post->post_type;
            $post_type_object = get_post_type_object($post_type);
            $can_publish = current_user_can($post_type_object->cap->publish_posts);
?>
<div class="submitbox" id="submitpost">
    <div id="minor-publishing">
        <div class="clear"></div>
    </div>
    <div id="major-publishing-actions">
        <div id="delete-action">
<?php
if ( current_user_can( "delete_post", $post->ID ) ) {
    if ( !EMPTY_TRASH_DAYS )
        $delete_text = __('Delete Permanently');
    else
        $delete_text = __('Move to Trash');
?>
            <a class="submitdelete deletion" href="<?php echo get_delete_post_link($post->ID); ?>"><?php echo $delete_text; ?></a><?php
} ?>
        </div>
        <div id="publishing-action">
            <span class="spinner"></span>
<?php
if ( !in_array( $post->post_status, array('publish', 'future', 'private') ) || 0 == $post->ID ) {
    if ( $can_publish ) :
        if ( !empty($post->post_date_gmt) && time() < strtotime( $post->post_date_gmt . ' +0000' ) ):
?>
            <input name="original_publish" type="hidden" id="original_publish" value="<?php esc_attr_e('Schedule') ?>" />
            <?php submit_button( __( 'Schedule' ), 'primary button-large', 'publish', false, array( 'accesskey' => 'p' ) ); ?>
<?php   else : ?>
            <input name="original_publish" type="hidden" id="original_publish" value="<?php esc_attr_e('Publish') ?>" />
            <?php submit_button( __( 'Publish' ), 'primary button-large', 'publish', false, array( 'accesskey' => 'p' ) ); ?>
<?php   endif;
    else : ?>
            <input name="original_publish" type="hidden" id="original_publish" value="<?php esc_attr_e('Submit for Review') ?>" />
            <?php submit_button( __( 'Submit for Review' ), 'primary button-large', 'publish', false, array( 'accesskey' => 'p' ) ); ?>
<?php
    endif;
} else { ?>
            <input name="original_publish" type="hidden" id="original_publish" value="<?php esc_attr_e('Update') ?>" />
            <input name="save" type="submit" class="button button-primary button-large" id="publish" accesskey="p" value="<?php esc_attr_e('Update') ?>" />
<?php
}
?>
        </div>
        <div class="clear"></div>
    </div>
</div>

<?php
        }

        private function getTaxonomiesList( ){
            $taxonomies = array( );
            foreach( get_taxonomies( null, 'objects' ) as $tax ){
                $label = ( $tax->labels && $tax->labels->name ? $tax->labels->name : $tax->label );
                $taxonomies[ $tax->name ] = $label;
            }

            return $taxonomies;
        }

        private function buildForm( $form, $values, $formID = 'dmptm' ){
            foreach( $form as $name => $field ){
                $id = $formID . '_' . $name;
                if( $field[ 'type' ] != 'advanced' ){
                    echo '<tr>';
                    echo '<td class="left">';
                    echo '<label for="' . $id . '">' . $field[ 'label' ] . '</label>';
                    if( keyCheck( $field, 'desc' ) )
                        echo '<p class="description">' . $field[ 'desc' ] . '</p>';
                    echo '</td>';
                    echo '<td class="right">';
                    switch( $field[ 'type' ] ){
                        case 'radio':
                            if( $field[ 'display' ] == 'inline' )
                                echo '<ul class="inline-checkbox-list">';
                            else
                                echo '<ul class="checkbox-list">';
                            foreach( $field[ 'options' ] as $n => $option ){
                                $checked = ( isset( $values[ $name ] ) && $option[ 'save' ] === $values[ $name ] ? ' checked' : '' );
                                echo '<li>';
                                echo '<label for="' . $id . '_' . $n . '">';
                                echo '<input type="radio" id="' . $id . '_' . $n . '" name="' . $id . '" value="' . $n . '"' . $checked . ' />';
                                echo $option[ 'label' ];
                                echo '</label>';
                                echo '</li>';
                            }
                            echo '</ul>';
                            break;
                        case 'checkbox':
                            $checked = ( keyCheck( $values, $name ) ? ' checked' : '' );
                            echo '<input type="checkbox" id="' . $id . '" name="' . $id . '" value="true"' . $checked . ' />';
                            break;
                        case 'select':
                            echo '<select id="' . $id . '" name="' . $id . '">';
                            foreach( $field[ 'values' ] as $value => $label ){
                                $selected = ( $values[ $name ] && $values[ $name ] == $value ? ' selected' : '' );
                                echo '<option value="' . $value . '"' . $selected . '>' . $label . '</option>';
                            }
                            echo '</select>';
                            break;
                        case 'multi_select':
                            echo '<select id="' . $id . '" name="' . $id . '[]" multiple>';
                            foreach( $field[ 'values' ] as $value => $label ){
                                $selected = ( keyCheck( $values, $name ) && is_array( $values[ $name ] ) && in_array( $value, $values[ $name ] ) ? ' selected' : '' );
                                echo '<option value="' . $value . '"' . $selected . '>' . $label . '</option>';
                            }
                            echo '</select>';
                            break;
                        case 'editable_select':
                            $found = false;
                            $val = ( isset( $values[ $name ] ) ? $values[ $name ] : '' );
                            echo '<select id="' . $id . '" name="' . $id . '" data-role="editable-select">';
                            foreach( keyCheck( $field, 'values', array( ) ) as $value => $label ){
                                $selected = ( $value == $val ? 'selected' : '' );
                                if( $selected ) $found = true;
                                echo '<option value="' . $value . '"' . $selected . '>' . $label . '</option>';
                            }

                            echo '<option value="' . ( $found || !$val ? '' : $val ) . '"' . ( !$found && $val ? ' selected' : '' ) . '>';
                            echo ( $found || !$val ? '[Custom]' : '- ' . $val . ' -' );
                            echo '</option>';
                            echo '</select>';
                            break;
                        case 'repeat':
                            echo '<div id="' . $id . '_repeat" data-role="repeat"' . ( isset( $field[ 'editKey' ] ) && $field[ 'editKey' ] ? ' data-edit-keys="true"' : '' ) . ' class="repeat">';
                            if( is_array( keyCheck( $values, $name ) ) && count( $values[ $name ] ) ){
                                $index = 0;
                                foreach( $values[ $name ] as $key => $option ){
                                    echo '<div id="' . $id . '_entry_' . $index . '" class="repeat-entry dm-columns">';
                                    if( isset( $field[ 'editKey' ] ) && $field[ 'editKey' ] ){
                                        echo '<div class="left-col">';
                                        echo '<input type="text" id="' . $id . '_' . $index . '_key" name="' . $id . '[][key]" value="' . $key . '" placeholder="Post Capability" />';
                                        echo '</div>';
                                        echo '<div class="right-col">';
                                        echo '<input type="text" id="' . $id . '_' . $index . '_value" name="' . $id . '[][value]" value="' . $option . '" placeholder="User Capability" />';
                                        echo '</div>';
                                    } else{
                                        echo '<div class="single-col">';
                                        echo '<input type="text" id="' . $id . '_' . $index . '" name="' . $id . '[]" placeholder="' . $field[ 'label' ] . '" value="' . $option . '" />';
                                        echo '</div>';
                                    }
                                    echo '<button type="button" class="button remove-entry" data-role="remove-repeat-entry" data-target="#' . $id . '_entry_' . $index . '" tabindex="-1">';
                                    echo '<span class="dashicons dashicons-no"></span>';
                                    echo '</button>';
                                    echo '</div>';
                                    $index++;
                                }
                            }
                            echo '<div class="no-entries"' . ( is_array( keyCheck( $values, $name ) ) && count( $values[ $name ] ) ? ' style="display:none"' : '' ) . '>' . $field[ 'noneText' ] . '</div>';
                            echo '<input type="button" class="button add-entry" data-role="add-repeat-entry" data-name="' . $id . '" value="' . $field[ 'addText' ] . '" />';
                            echo '</div>';
                            break;
                        case 'callback':
                            call_user_func_array( $field[ 'render' ], array( $field, $id, $name, $form, $values, $formID ) );
                            break;
                        default:
                            echo '<input type="text" id="' . $id . '" name="' . $id . '"' . ( keyCheck( $values, $name ) && $values[ $name ] ? ' value="' . $values[ $name ] . '"' : '' ) . ' />';
                            break;
                    }
                    echo '</td>';
                    echo '</tr>';
                } else{
                    echo '<tr>';
                    echo '<td class="left heading">';
                    echo '<h3 class="heading">' . $field[ 'label' ] . '</h3>';
                    echo '</td>';
                    echo '<td class="right heading">';
                    echo '</td>';
                    echo '</tr>';
                    $this->buildForm( keyCheck( $field, 'fields' ), ( keyCheck( $values, $name ) ? $values[ $name ] : array( ) ), $id );
                }
            }
        }

        public function renderCapabilityType( $field, $id, $name, $form, $values, $formID ){
            $value = keyCheck( $values, $name, array( 'single' => 'post', 'bulk' => 'posts' ) );
            echo '<div id="' . $id . '" class="repeat">';
            echo '<div class="dm-columns">';
            echo '<div class="single-col">';
            echo '<label for="' . $id . '_single">Single Post Capabilities</lable>';
            echo '</div>';
            echo '<div class="single-col">';
            echo '<input type="text" id="' . $id . '_single" name="' . $id . '[single]" value="' . $value[ 'single' ] . '" />';
            echo '</div>';
            echo '<div class="single-col">';
            echo '<label for="' . $id . '_bulk">Bulk Post Capabilities</lable>';
            echo '</div>';
            echo '<div class="single-col">';
            echo '<input type="text" id="' . $id . '_bulk" name="' . $id . '[bulk]" value="' . $value[ 'bulk' ] . '" />';
            echo '</div>';
            echo '</div>';
            echo '</div>';
        }

        public function renderMenuIcons( $field, $id, $name, $form, $values, $formID ){
            $dashicons = array(
                'dashicons-menu','dashicons-admin-site','dashicons-dashboard','dashicons-admin-media','dashicons-admin-page','dashicons-admin-comments','dashicons-admin-appearance','dashicons-admin-plugins','dashicons-admin-users',
                'dashicons-admin-tools','dashicons-admin-settings','dashicons-admin-network','dashicons-admin-generic','dashicons-admin-home','dashicons-admin-collapse','dashicons-admin-links','dashicons-format-links','dashicons-admin-post',
                'dashicons-format-standard','dashicons-format-image','dashicons-format-gallery','dashicons-format-audio','dashicons-format-video','dashicons-format-chat','dashicons-format-status','dashicons-format-aside',
                'dashicons-format-quote','dashicons-welcome-write-blog','dashicons-welcome-edit-page','dashicons-welcome-add-page','dashicons-welcome-view-site','dashicons-welcome-widgets-menus','dashicons-welcome-comments',
                'dashicons-welcome-learn-more','dashicons-image-crop','dashicons-image-rotate-left','dashicons-image-rotate-right','dashicons-image-flip-vertical','dashicons-image-flip-horizontal','dashicons-undo','dashicons-redo',
                'dashicons-editor-bold','dashicons-editor-italic','dashicons-editor-ul','dashicons-editor-ol','dashicons-editor-quote','dashicons-editor-alignleft','dashicons-editor-aligncenter','dashicons-editor-alignright',
                'dashicons-editor-insertmore','dashicons-editor-spellcheck','dashicons-editor-distractionfree','dashicons-editor-expand','dashicons-editor-contract','dashicons-editor-kitchensink','dashicons-editor-underline',
                'dashicons-editor-justify','dashicons-editor-textcolor','dashicons-editor-paste-word','dashicons-editor-paste-text','dashicons-editor-removeformatting','dashicons-editor-video','dashicons-editor-customchar',
                'dashicons-editor-outdent','dashicons-editor-indent','dashicons-editor-help','dashicons-editor-strikethrough','dashicons-editor-unlink','dashicons-editor-rtl','dashicons-editor-break','dashicons-editor-code',
                'dashicons-editor-paragraph','dashicons-align-left','dashicons-align-right','dashicons-align-center','dashicons-align-none','dashicons-lock','dashicons-calendar','dashicons-calendar-alt','dashicons-visibility',
                'dashicons-post-status','dashicons-edit','dashicons-post-trash','dashicons-trash','dashicons-external','dashicons-arrow-up','dashicons-arrow-down','dashicons-arrow-left','dashicons-arrow-right','dashicons-arrow-up-alt',
                'dashicons-arrow-down-alt','dashicons-arrow-left-alt','dashicons-arrow-right-alt','dashicons-arrow-up-alt2','dashicons-arrow-down-alt2','dashicons-arrow-left-alt2','dashicons-arrow-right-alt2','dashicons-leftright',
                'dashicons-sort','dashicons-randomize','dashicons-list-view','dashicons-excerpt-view','dashicons-excerpt-view','dashicons-grid-view','dashicons-hammer','dashicons-art','dashicons-migrate','dashicons-performance',
                'dashicons-universal-access','dashicons-universal-access-alt','dashicons-tickets','dashicons-nametag','dashicons-clipboard','dashicons-heart','dashicons-megaphone','dashicons-schedule','dashicons-wordpress',
                'dashicons-wordpress-alt','dashicons-pressthis','dashicons-update','dashicons-screenoptions','dashicons-info','dashicons-cart','dashicons-feedback','dashicons-cloud','dashicons-translation','dashicons-tag','dashicons-category',
                'dashicons-archive','dashicons-tagcloud','dashicons-text','dashicons-media-archive','dashicons-media-audio','dashicons-media-code','dashicons-media-default','dashicons-media-document','dashicons-media-interactive',
                'dashicons-media-spreadsheet','dashicons-media-text','dashicons-media-video','dashicons-playlist-audio','dashicons-playlist-video','dashicons-controls-play','dashicons-controls-pause','dashicons-controls-forward',
                'dashicons-controls-skipforward','dashicons-controls-back','dashicons-controls-skipback','dashicons-controls-repeat','dashicons-controls-volumeon','dashicons-controls-volumeoff','dashicons-yes','dashicons-no','dashicons-no-alt',
                'dashicons-plus','dashicons-plus-alt','dashicons-minus','dashicons-dismiss','dashicons-marker','dashicons-star-filled','dashicons-star-half','dashicons-star-empty','dashicons-flag','dashicons-share','dashicons-share1',
                'dashicons-share-alt','dashicons-share-alt2','dashicons-twitter','dashicons-rss','dashicons-email','dashicons-email-alt','dashicons-facebook','dashicons-facebook-alt','dashicons-networking','dashicons-googleplus',
                'dashicons-location','dashicons-location-alt','dashicons-camera','dashicons-images-alt','dashicons-images-alt2','dashicons-video-alt','dashicons-video-alt2','dashicons-video-alt3','dashicons-vault','dashicons-shield',
                'dashicons-shield-alt','dashicons-sos','dashicons-search','dashicons-slides','dashicons-analytics','dashicons-chart-pie','dashicons-chart-bar','dashicons-chart-line','dashicons-chart-area','dashicons-groups',
                'dashicons-businessman','dashicons-id','dashicons-id-alt','dashicons-products','dashicons-awards','dashicons-forms','dashicons-testimonial','dashicons-portfolio','dashicons-book','dashicons-book-alt','dashicons-download',
                'dashicons-upload','dashicons-backup','dashicons-clock','dashicons-lightbulb','dashicons-microphone','dashicons-desktop','dashicons-tablet','dashicons-smartphone','dashicons-phone','dashicons-smiley','dashicons-index-card',
                'dashicons-carrot','dashicons-building','dashicons-store','dashicons-album','dashicons-palmtree','dashicons-tickets-alt','dashicons-money'
            );

            echo '<input type="text" id="' . $id . '" name="' . $id . '" value="' . keyCheck( $values, $name, 'dashicons-admin-generic' ) . '" />';
            echo '<div id="dashicons-list">';
            foreach( $dashicons as $dashicon ){
                echo '<a class="button button-default" data-icon="' . $dashicon . '" title="' . $dashicon . '">';
                echo '<span class="dashicons ' . $dashicon . '"></span></a>';
            }
            echo '</div>';
        }
    }

    add_action( 'dm_init', 'DM_PostTypeManager::getInstance' );
    add_action( 'dm_init', 'DM_PostTypeAdmin::getInstance' );

    if( !function_exists( 'array_replace' ) ){
        // Backwards compatibility for PHP 5.2 and older.
        function array_replace( /* variable number of arguments */ ){
            $result = array( );
            foreach( func_get_args( ) as $arg ){
                foreach( $arg as $key=>$value ){
                    $result[ $key ] = $value;
                }
            }

            return $result;
        }
    }
