<?php
    /**
     *  Module Name: DM Smart Thumbnails
     *  Version:     v1.0
     *  Description: A utility that attempts to find a featured images for every post/custom post type.
     *  Author:      Distill Mill LLC
     *  Author URI:  http://distillmill.com
     */

    add_action( 'init', 'DM_SmartThumbnail::getInstance' );

    class DM_SmartThumbnail{
        protected static $instance;
        private $attachments = array( );

        public static function getInstance( ){
            if( !isset( self::$instance ) )
                new self;

            return self::$instance;
        }

        public static function call( $method ){
            return array( self::getInstance( ), $method );
        }

        public function __construct( ){
            self::$instance = $this;
            add_filter( 'get_post_metadata',     self::call( 'customGetPostMeta' ),    10, 4 );
            add_filter( 'delete_post_metadata',  self::call( 'customDeletePostMeta' ), 10, 5 );
            add_filter( 'update_post_metadata',  self::call( 'customUpdatePostMeta' ), 10, 5 );
            add_filter( 'add_post_metadata',     self::call( 'customAddPostMeta' ),    10, 5 );
            add_filter( 'admin_enqueue_scripts', self::call( 'localize' ) );
        }

        public function localize( ){
            // Ensure the text is localized... for some reason it doesn't always
            // seem to be loaded which results in the front end failing to
            // dynamically remove the image from the page.
            $text = array(
                'setThumbnail' => __( 'Use as featured image' ),
                'saving' => __( 'Saving...' ),
                'error' => __( 'Could not set that as the thumbnail image. Try a different attachment.' ),
                'done' => __( 'Done' )
            );

            wp_localize_script( 'dm-smart-thumbs','setPostThumbnailL10n', $text );
        }

        public function customGetPostMeta ( $result, $objectID, $metakey, $single ){
            global $wpdb;
            if( $metakey == '_thumbnail_id' ){
                $screen = ( function_exists( 'get_current_screen' ) ? get_current_screen( ) : null );
                // If in the admin make sure we're not on a post edit screen.
                // This is so that the featured image will display whatever the
                // user has set or nothing if they haven't set anything.
                if( metadata_exists( 'post', $objectID, '_dm_thumbnail_id' ) ) {
                    $result = get_metadata( 'post', $objectID, '_dm_thumbnail_id', $single );
                } elseif( !is_admin( ) || $screen->base != 'post' ) {
                    if( $this->getWooCommerceGalleryImage( $objectID ) !== false ) {
                        $result = intval( $this->getWooCommerceGalleryImage( $objectID ) );
                    } elseif( $this->getDM_ACF_FeaturedImages( $objectID ) !== false ) {
                        $result = intval( $this->getDM_ACF_FeaturedImages( $objectID ) );
                    } elseif( count( $this->getAttachedImages( $objectID ) ) > 0 ){
                        $attachments = $this->getAttachedImages( $objectID );
                        $result = $attachments[ 0 ]->ID;
                    } else {
                        // Ok heres where things get interesting...
                        // Gonna try and find an image we can use by searching
                        // through the posts content. This can be time consuming
                        // (well at least more time consuming than most things)
                        // so we will cache the result for 24 hours in a
                        // transient.
                        // If the transient doesn't exist (either never created
                        // or expired) find all images in the content, loop
                        // through them until we find one we can find in the
                        // database. Once found cache it in the transient.
                        $post = get_post( $objectID );
                        $imageID = get_transient( 'dm-smart-thumbs-content-image-' . $objectID );
                        if( !$imageID ){  // Transient not set or it expired
                            $data = array( );
                            preg_match_all( "/\<img [^\>]+src=[\'\"]([^\'\"]+)/", $post->post_content, $data );
                            if( isset( $data[ 1 ] ) ){  // We have images in the content
                                $uploads = wp_upload_dir( );
                                foreach( $data[ 1 ] as $imageURL ){
                                    // Is the image in the site's uploads directory
                                    if( substr( $imageURL, 0, strlen( $uploads[ 'baseurl' ] ) ) == $uploads[ 'baseurl' ] ){
                                        // This is the part of the url that WordPress stores with the attachment.
                                        $storedURL = substr( $imageURL, strlen( $uploads[ 'baseurl' ] ) + 1 );
                                        // Any attachments that are associated with this image.
                                        $attachments = $wpdb->get_results( "SELECT post_id FROM $wpdb->postmeta WHERE meta_value='$storedURL'" );
                                        if( count( $attachments ) > 0 ){
                                            // Get the attachment ID
                                            $result = $imageID = $attachments[ 0 ]->post_id;
                                            // Cache the attachment ID
                                            set_transient( 'dm-smart-thumbs-content-image-' . $objectID, $imageID, 60*60*24 );
                                            // Stop looking because we found it
                                            break;
                                        }
                                    }
                                }
                            }
                        } else {
                            $result = $imageID;
                        }

                        // If there still isn't an image found use the
                        // appropriate placeholder.
                        if( !$imageID ){
                            $postType = $post->post_type;
                            $image = null;
                            $placeholders = DM( )->getField( 'placeholder_images', 'option' );
                            if( $placeholders ){
                                foreach( $placeholders as $placeholder ){
                                    if( $placeholder[ 'post_type' ] == $postType ){
                                        $image = $placeholder[ 'image' ][ 'id' ];
                                        break;  // We found what we needed so exit
                                    } elseif( $placeholder[ 'post_type' ] == 'default' )
                                        $image = $placeholder[ 'image' ][ 'id' ];
                                    elseif( is_null( $default ) )  // Let's grab the first thing we find and save it so we have something for sure
                                        $image = $placeholder[ 'image' ][ 'id' ];
                                }
                            }

                            if( !is_null( $image ) )
                                $result = $image;
                        }
                    }
                } else {
                    $result = '';
                }
            }

            return $result;
        }

        public function getMatchingPlaceholder( $postType ){
            $image = null;
            $default = null;

            $placeholders = DM( )->getField( 'placeholder_images', 'option' );
            if( $placeholders ){
                foreach( $placeholders as $placeholder ){
                    if( $placeholder[ 'post_type' ] == $postType ){
                        $image = $placeholder[ 'image' ][ 'id' ];
                        break;  // We found what we needed so exit
                    }
                    elseif( $placeholder[ 'post_type' ] == 'default' )
                        $default = $placeholder[ 'image' ][ 'id' ];
                    elseif( is_null( $default ) )  // Let's grab the first thing we find and save it so we have something for sure
                        $image = $placeholder[ 'image' ][ 'id' ];
                }
            }

            return ( $image ? $image : $default );
        }

        public function customDeletePostMeta( $result, $objectID, $metakey, $metaValue, $deleteAll ){
            // If deleting the thumbnail go ahead and remove it from the custom storage instead
            if( $metakey == '_thumbnail_id' ){
                delete_metadata( 'post', $objectID, '_dm_thumbnail_id', $metaValue, $deleteAll );
            }

            return $result;
        }

        public function customUpdatePostMeta( $result, $objectID, $metakey, $metaValue, $prevValue ){
            // If updating the thumbnail go ahead and update the custom storage instead
            if( $metakey == '_thumbnail_id' ){
                update_metadata( 'post', $objectID, '_dm_thumbnail_id', $metaValue, $prevValue );
            }

            return $result;
        }

        public function customAddPostMeta( $result, $objectID, $metakey, $metaValue, $unique ){
            // If adding a thumbnail go ahead and add it to the custom storage instead
            if( $metakey == '_thumbnail_id' ){
                $result = add_metadata( 'post', $objectID, '_dm_thumbnail_id', $metaValue, $unique );
            }

            return $result;
        }

        private function getWooCommerceGalleryImage( $objectID ){
            if( class_exists( 'WC_Product_Factory' ) ){
                $post = get_post( $objectID );
                if( $post->post_type == 'product' && $post->post_status == 'publish' ){  // Bombs hard for non or unpublished products if we don't make this check
                    $pf = new WC_Product_Factory( );

                    //$attachments = $pf->get_product( $post )->get_gallery_attachment_ids( );
                    /*
                    if( !empty( $attachments ) ) {
                        return $attachments[ 0 ];
                    }
                    */
                }
            }

            return false;
        }

        private function getDM_ACF_FeaturedImages( $objectID ){
            $result = DM()->getField( 'gallery', $objectID );
            if( $result )
                return $result[ 0 ][ 'id' ];
            $result = DM()->getField( 'dm_wide_format_image', $objectID );
            if( $result )
                return $result[ 'id' ];

            $result = DM()->getField( 'dm_tall_format_image', $objectID );
            if( $result )
                return $result[ 'id' ];

            return false;
        }

        private function getAttachedImages( $postID ){
            if( !isset( $this->attachments[ $postID ] ) ){
                $this->attachments[ $postID ] = get_posts(
                    array(
                        'post_parent'    => $postID,
                        'post_type'      => 'attachment',
                        'numberposts'    => -1,
                        'post_mime_type' => 'image',
                        'orderby'        => 'post_date',
                        'order'          => 'ASC'
                    )
                );
            }

            return $this->attachments[ $postID ];
        }
    }

    // Theme tags
    function get_post_thumbnail_url( $post=null ){
        if( is_null( $post ) )
            $post = get_the_ID( );
        elseif( $post instanceof WP_Post )
            $post = $post->ID;

        if( !$post )
            return wp_get_attachment_url( DM_SmartThumbnail::getInstance( )->getMatchingPlaceholder( 'not_found' ) );

        return wp_get_attachment_url( get_post_thumbnail_id( $post ) );
    }
