<?php
    /**
     *  Module Name: DM Disable Comments
     *  Version:     v1.0
     *  Description: A utility that disable comments.
     *  Author:      Distill Mill LLC
     *  Author URI:  http://distillmill.com
     */

    class DM_DisableComments{
        private static $instance;
        private $settings;

        public static function getInstance( ){
            if( !isset( self::$instance ) ){
                self::$instance = new self;
                self::$instance->init( );
            }

            return self::$instance;
        }

        public static function call( $method ){
            return array( self::getInstance( ), $method );
        }

        public function init( ){
            // Tells everything in the loop whether comments are open.
            add_filter( 'comments_open', self::call( 'closeComments' ), 10, 2 );

            // Reflects the DM setting for the post post type in the
            // Settings>Discussion page.
            add_filter( 'option_default_comment_status', self::call( 'allowingComments' ) );
            
            // When the post is being re-added to the database with updates this
            // is where we check if the post is overriding the DM setting for
            // its post type.
            add_filter( 'wp_insert_post_data', self::call( 'updatePostIndex' ), 10, 2 );

            // Reflects the Settings>Discussion comments setting changes in the
            // DM settings for the post post type.
            add_action( 'update_option', self::call( 'updateComments' ), 10, 3 );

            // These filter the post object as it is being created but only in the admin.
            add_filter( 'edit_post_comment_status', self::call( 'filterCommentStatus' ), 10, 2 );
            add_filter( 'post_comment_status', self::call( 'filterCommentStatus' ), 10, 2 );

            if( is_admin( ) )
                DM( )->admin->addTabMetaBox( 'Comments', array( 'general_settings' ), self::call( 'showCommentsSettings' ), self::call( 'saveCommentsSettings' ) );
        }

        // Filters the site default setting to match the DM setting for the post
        // post type.
        public function allowingComments( $value ){
            return ( $this->isPostTypeEnabled( 'post' ) ? 'open' : 'closed' );
        }

        // Filters the comment status of a post but only when checked using the
        // comments_open function.
        public function closeComments( $open, $postID ){
            $post = get_post( $postID );
            if( $post ){
                $index = $this->getPostTypeIndex( $post->post_type );
                if( $index && (int)get_post_meta( $post->ID, 'dm-disable-comments-override', true ) == $index )
                    return $post->comment_status == 'open';
                else
                    return $this->isPostTypeEnabled( $post->post_type );
            }

            return $open;
        }

        // Filters the post object as it is being created to ensure that posts
        // respect the DM setting for the posts post type. This only works if
        // post->filter is called (so normally not on the front end).
        public function filterCommentStatus( $value, $postID ){
            $post = get_post( $postID );
            $oldValue = $value;
            if( $post ){
                $index = $this->getPostTypeIndex( $post->post_type );
                if( !$index || (int)get_post_meta( $post->ID, 'dm-disable-comments-override', true ) != $index )
                    $value = ( $this->isPostTypeEnabled( $post->post_type ) ? 'open' : 'closed' );
            }

            return $value;
        }

        // Returns the current override index for a given post type. Posts must
        // have an override index that matches their post types inorder to
        // override the post type setting.
        public function getPostTypeIndex( $postType ){
            $settings = $this->getSettings( );
            if( !array_key_exists( $postType, $settings ) ){
                $this->enablePostType( $postType, false );
                $settings = $this->getSettings( );
            }
            return $settings[ $postType ][ 'index' ];
        }

        // Returns the DM setting for the given post type.
        // Note: This returns either true or false not open or closed.
        public function isPostTypeEnabled( $postType ){
            $settings = $this->getSettings( );
            if( is_array( $settings ) && array_key_exists( $postType, $settings ) )
                return $settings[ $postType ][ 'enabled' ];
            return false;
        }

        // Saves the meta box.
        public function saveCommentsSettings( ){
            $postTypes = ( isset( $_POST[ 'dm-disable-comments' ] ) ? $_POST[ 'dm-disable-comments' ] : array( ) );
            $settings = $this->getSettings( );
            // Loop through all the post types we already have settings for.
            foreach( $settings as $postType=>$setting ){
                $this->enablePostType( $postType, in_array( $postType, $postTypes ), false );
                if( in_array( $postType, $postTypes ) )
                    unset( $postTypes[ array_search( $postType, $postTypes ) ] );
            }
            // Loop through any new post types we don't already have settings for.
            foreach( $postTypes as $postType ){
                $this->enablePostType( $postType, true, false );
            }
            $this->saveSettings( );
        }

        // Shows the DM settings in a meta box.
        public function showCommentsSettings( ){
            echo '<p>Check to enable commenting for the post type.</p>';
            foreach( get_post_types( false, 'objects' ) as $slug=>$postType ){
                if( post_type_supports( $slug, 'comments' ) ){
                    echo '<label>';
                    echo '<input type="checkbox" name="dm-disable-comments[]" value="' . $slug . '"';
                    if( $this->isPostTypeEnabled( $slug ) )
                        echo ' checked';
                    echo ' /> ';
                    echo $postType->label;
                    echo '</label>';
                    echo '<br />';
                }
            }
        }

        // Updates the DM setting for the post post type to match changes to the
        // site comments default setting.
        public function updateComments( $option, $old, $new ){
            if( $option == 'default_comment_status' )
                $this->enablePostType( 'post', $new == 'open' );
        }

        // This index is used to activate overrides. If the index matches the
        // current post type index then the posts setting is used.
        public function updatePostIndex( $data, $arr ){
            $post = get_post( $arr[ 'ID' ] );
            $index = get_post_meta( $post->ID, 'dm-disable-comments-override', true );
            $ptIndex = $this->getPostTypeIndex( $post->post_type );
            if( !$index || $index != $ptIndex || $data[ 'comment_status' ] != $post->comment_status ){
                $index = $this->getPostTypeIndex( $post->post_type );
                update_post_meta( $post->ID, 'dm-disable-comments-override', $index );
            }
            return $data;
        }

        // Enables a post type and makes the necessary DB changes.
        private function enablePostType( $postType, $enabled=true, $save=true ){
            $settings = $this->getSettings( );
            if( array_key_exists( $postType, $settings ) ){
                if( $enabled != $settings[ $postType ][ 'enabled' ] )
                    $settings[ $postType ][ 'index' ]++;
                $settings[ $postType ][ 'enabled' ] = $enabled;
            } else{
                $settings[ $postType ] = array(
                    'enabled' => $enabled,
                    'index'   => 1
                );
            }

            if( $save )
                $this->saveSettings( $settings );
            else
                $this->settings = $settings;
        }

        // Gets the settings
        private function getSettings( ){
            if( !isset( $this->settings ) )
                $this->settings = get_option( 'dm-disable-comments-settings' );

            return $this->settings;
        }

        // Saves changes to the DB.
        private function saveSettings( $settings=null ){
            if( $settings )
                $this->settings = $settings;

            update_option( 'dm-disable-comments-settings', $this->settings );
        }
    }

    add_action( 'dm_init', 'DM_DisableComments::getInstance' );
