<?php
    /**
     *  Module Name: DM Themes
     *  Version:     v1.0
     *  Description: A utility that makes theme functions easier to manage and provides some boilerplate methods.
     *  Author:      Distill Mill LLC
     *  Author URI:  http://distillmill.com
     */

    // Handles the loading of theme classes
    class DM_ThemesCore{
        private static $instance;
        private static $themes = array( );
        private $coreKey;

        public static function getInstance( $dm=null, $coreKey=null ){
            if( !isset( self::$instance ) ){
                self::$instance = new self;
                self::$instance->coreKey = $coreKey;
                self::$instance->load( );
            }

            return self::$instance;
        }

        // A fancy method to allow for easily providing any public method of this
        // class as a callback. It works by ensuring that there is an instance then
        // passing a reference to that instance. It is necessary to pass a reference
        // because if this method is called while the constructor is running there
        // will not be an instance.
        // NOTE: This does not call the method it just provides a valid callback.
        public static function call( $method ){
            return array( self::getInstance( ), $method );
        }

        public function load( ){
            DM( )->registerMethod( 'registerTheme', self::call( 'registerTheme' ) );
            add_action( 'after_setup_theme', self::call( 'loadThemeClasses' ) );

            // Need to accept 2 arguments because since WordPress 4.2 the post's
            // ID is passed.
            add_filter( 'the_title', self::call( 'pageTitle' ), 10, 2 );
        }

        public function loadThemeClasses( ){
            foreach( get_declared_classes( ) as $class ){
                if( is_subclass_of( $class, 'DM_Theme' ) )
                    self::$themes[ ] = new $class;
            }
        }

        // A filter for titles that applies the dm_main_title filter if it
        // determines that the current post is the page's main post. If it
        // determines that the current post is one of many posts in the main
        // loop it will apply dm_loop_title.
        // NOTE: Must default $postID because prior to WordPress 4.2 the post's
        //       ID is not passed.
        public function pageTitle( $title, $postID=null ){
            if( in_the_loop( ) ){
                if( is_main_query( ) ){
                    if( is_single( ) || is_page( ) ){
                        global $wp_the_query;
                        if( is_null( $postID ) || $postID == $wp_the_query->queried_object_id ){
                            $title = apply_filters( 'dm_main_title', $title );
                        }
                    } else{
                        $title = apply_filters( 'dm_loop_title', $title );
                    }
                }
            }

            return $title;
        }

        public function registerTheme( $name, $theme ){
            if( is_subclass_of( $theme, 'DM_Theme' ) )
                DM( )->registerMember( $name, $theme, $this->coreKey );
        }
    }

    // Base class for any theme classes that would benefit from integration with
    // DM Utils and the base theme boilerplate methods.
    class DM_Theme{
        public static $instances;
        private $excerpters = array( );

        public function call( $method ){
            return array( $this, $method );
        }

        public function endExcerpter( ){
            array_pop( $this->excerpters );
            if( !$this->excerpters ){
                remove_filter( 'excerpt_length', $this->call( 'excerptLength' ), 100 );
                remove_filter( 'excerpt_more', $this->call( 'excerptMore' ), 100 );
            }
        }

        public function endAllExcerpters( ){
            $this->excerpters = array( );
            remove_filter( 'excerpt_length', $this->call( 'excerptLength' ), 100 );
            remove_filter( 'excerpt_more', $this->call( 'excerptMore' ), 100 );
        }

        public function directionsLink( $link, $text = null, $class = '' ){
        if( !$text ) $text = 'Directions &#62;';
            return '<a href="http://maps.google.com/maps?saddr=current+location'
                 . '&daddr=' . urlencode( $link ) . '&t=m&z=3" rel="directions" '
                 . 'class="directions-link ' . $class . '" target="_blank">'
                 . $text . '</a>';
        }

        public function emailLink( $link, $text = null, $class = '' ){
            if( !$text ) $text = 'Email &#62;';
            return '<a href="mailto:' . $link . '" rel="email" class="email-link '
                 . $class . '">' . $text . '</a>';
        }

        public function excerptLength( $length ){
            $settings = $this->findExcerptSettings( );
            if( !is_null( $settings[ 'length' ] ) )
                $length = $settings[ 'length' ];

            return $length;
        }

        public function excerptMore( $more ){
            $settings = $this->findExcerptSettings( );
            if( !is_null( $settings[ 'more' ] ) )
                $more = $settings[ 'more' ];

            return $more;
        }

        // Gets the URL for the given image with the given settings.
        public function sizeImage( $imageURL, $width, $height = null, $anchor = null ){
            if( !isset( $imageURL ) && file_exists( get_template_directory( ) . '/img/placeholder.png' ) )
                $imageURL = DM( )->getTemplateURL( ) . '/img/placeholder.png';
            
            // If there is an image URL and any of the settings are set then create
            // a timthumb URL.
            if( $imageURL && ( $width || $height || $anchor ) ){
                $imageURL = TIMTHUMB_URL . '?src=' . $imageURL;
                if( $width )
                    $imageURL .= '&w=' . $width;
                if( $height )
                    $imageURL .= '&h=' . $height;
                if( $anchor )
                    $imageURL .= '&a=' . $anchor;
            }

            return ( $imageURL ? $imageURL : '' );  // Make sure a string is being returned
        }

        public function startExcerpter( $length, $more ){
            if( !$this->excerpters ){
                add_filter( 'excerpt_length', $this->call( 'excerptLength' ), 100 );
                add_filter( 'excerpt_more', $this->call( 'excerptMore' ), 100 );
            }

            $this->excerpters[ ] = array(
                'length' => $length,
                'more'   => $more
            );
        }

        public function telephoneLink( $link, $text = null, $class = '' ){
            if( !$text ) $text = 'Call &#62;';
            return '<a href="tel:+1'
                 . substr( preg_replace( '/[^0-9]/', '', $link ), 0, 10 )
                 . '" rel="call" class="telephone-link ' . $class . '">'
                 . $text . '</a>';
        }

        public function wrapTheTitle( $tag, $attrs=false ){
            echo '<' . $tag;
            if( $attrs ){
                foreach( $attrs as $name => $value ){
                    echo ' ' . $name . '="' . $value . '"';
                }
            }
            echo '>';
            the_title( );
            echo '</' . $tag . '>';
        }

        private function findExcerptSettings( ){
            $length = null;
            $more = null;
            foreach( array_reverse( $this->excerpters ) as $excerpter ){
                if( is_null( $length ) )
                    $length = $excerpter[ 'length' ];

                if( is_null( $more ) )
                    $length = $excerpter[ 'more' ];

                if( !is_null( $length ) && !is_null( $more ) )
                    break;
            }

            return array(
                'length' => $length,
                'more'   => $more
            );
        }
    }

    add_action( 'dm_init', 'DM_ThemesCore::getInstance', 10, 2 );
