<?php
    /**
     * Plugin Name: Distill Mill Utils
     * Plugin URI: http://distillmill.com
     * Description: This mu-plugin handles the loading of modules and core functionality necessary for Distill Mill sites.
     * Version: 0.2
     * Author: Distill Mill
     * Author URI: http://distillmill.com
     * License: Copyright 2015, Distill Mill
     */

    define( 'DM_UTILS_FILE', __FILE__ );
    define( 'DM_UTILS_DIR', plugin_dir_path( __FILE__ ) );
    define( 'DM_UTILS_URL', plugin_dir_url( __FILE__ ) );

    require_once( '_dm-utils/functions.php' );  // Load useful functions
    if( is_admin( ) )
        require_once( '_dm-utils/admin/admin.php' );  // Load the admin

    class DM_Utils{
        private static $instance;
        private $modules = array( );  // NOTE: This is all components not just what is in the modules directory
        private $enabledModules = false;  // False will enable all modules, an array of module file paths will enable those modules
        private $loaded = false;
        private $methods = array( );
        private $coreKey;

        public static function getInstance( ){
            if( !isset( self::$instance ) ){
                self::$instance = new self;
                self::$instance->coreKey = rand( );
                self::$instance->load( );  // Load all modules and start any that are enabled
            }

            return self::$instance;
        }

        // A fancy method to allow for easily providing any public method of this
        // class as a callback. It works by ensuring that there is an instance then
        // passing a reference to that instance. It is necessary to pass a reference
        // because if this method is called while the constructor is running there
        // will not be an instance.
        // NOTE: This does not call the method it just provides a valid callback.
        public static function call( $method ){
            self::getInstance( );
            return array( &self::$instance, $method );
        }

        // Allows us to run methods registered with the core object in a nice
        // clean way that makes everything feel seamless.
        public function __call( $name, $arguments ){
            if( isset( $this->methods[ $name ] ) )
                return call_user_func_array( $this->methods[ $name ], $arguments );

            else
                throw new Exception( 'Call to undefined method DM_Utils::' . $name . '()' );
        }

        public function adminEnqueue( ){
            wp_register_style( 'dm_admin_css', DM_UTILS_URL . '_dm-utils/resources/css/admin.min.css', false, '1.0.0' );
            wp_register_script( 'dm_admin_js', DM_UTILS_URL . '_dm-utils/resources/js/admin.min.js', array( 'jquery' ), '1.0.0' );
        }

        // Returns module data using the same methods as the WordPress plugin
        // loader.
        public function getModuleData( $moduleFile ){
            if( isset( $this->modules[ $moduleFile ] ) ){
                $data = $this->modules[ $moduleFile ];
            } else{
                $defaults = array(
                    'Name' => 'Module Name',
                    'ModuleURI' => 'Module URI',
                    'ModuleHelpURI' => 'Module Help URI',
                    'Version' => 'Version',
                    'Description' => 'Description',
                    'Author' => 'Author',
                    'AuthorURI' => 'Author URI',
                    'TextDomain' => 'Text Domain',
                    'DomainPath' => 'Domain Path',
                    'Network' => 'Network'
                );

                $data = get_file_data( $moduleFile, $defaults, 'plugin', 'dm-module' );
            }

            return $data;
        }

        // Returns the home URL and caches it for later calls.
        public function getHomeURL( ){
            if( !isset( $this->homeURL ) )
                $this->homeURL = get_home_url( );

            return $this->homeURL;
        }

        // Returns the list of all modules that DM Utils has found.
        public function getModules( ){
            return $this->modules;
        }

        // Returns templates directory URL and caches it for later calls.
        public function getTemplateURL( ){
            if( !isset( $this->templateURL ) )
                $this->templateURL = get_template_directory_uri( );

            return $this->templateURL;
        }

        // Loads all modules in a directory and includes any that are enabled.
        public function loadDirectory( $path ){
            $subdir = @opendir( $path );
            while ( ( $subfile = readdir( $subdir ) ) !== false ) {
                if ( substr( $subfile, -4 ) == '.php' ){
                    $modulePath = $path . '/' . $subfile;
                    $this->modules[ $modulePath ] = $this->getModuleData( $modulePath );
                    if( is_array( $this->enabledModules ) && in_array( $modulePath, $this->enabledModules ) ){
                        include_once( $modulePath );
                        $this->modules[ $modulePath ][ 'enabled' ] = true;
                    } else
                        $this->modules[ $modulePath ][ 'enabled' ] = false;
                }
            }
            @closedir( $path );
        }

        // Allows core modules to register members on the core object.
        // ie. Module x could register member y on the core object which would
        // allow DM( )->y to access the value set by x.
        public function registerMember( $name, $value, $coreKey=null ){
            if( $this->coreKey === $coreKey || !$this->loaded )
                $this->$name = $value;
        }

        // Allows core modules to register methods on the core object.
        // ie. Module x could register method y on the core object which would
        // allow DM( )->y( ) to call the method on x.
        public function registerMethod( $name, $func, $coreKey=null ){
            if( $this->coreKey === $coreKey || !$this->loaded )
                $this->methods[ $name ] = $func;
        }

        // An abstract function caller that ensures the method exists before trying
        // to call it. It will accept any number of arguments. The first argument
        // needs to be a function name and the rest will be passed to that function
        // if it exists. The result of the function call will be returned if it
        // exists otherwise null will be returned.
        public function safeCall( /* variable number of arguments */ ){
            $args = func_get_args( );
            if( count( $args ) > 0 )
                return $this->safeCallArray( $args[ 0 ], array_slice( $args, 1 ) );

            return null;
        }

        // An abstract function caller that ensures the method exists before trying
        // to call it. The first argument needs to be a function name and the 
        // second should be an array of arguments. The result of the function call
        // will be returned if it exists otherwise null will be returned.
        public function safeCallArray( $func, $args ){
            if( function_exists( $func ) )
                return call_user_func_array( $func, $args );

            return null;
        }

        // Starts DM Utils loading all modules and triggering all necessary
        // hooks.
        private function load( ){
            $this->enabledModules = get_option( 'dm-enabled-modules' );
            $path = DM_UTILS_DIR;

            add_action( 'admin_enqueue_scripts', self::call( 'adminEnqueue' ), 9 );
            
            // Load and initialize all dm-modules
            $this->loadDirectory( $path . '_modules' );
            do_action( 'dm_init', $this, $this->coreKey );  // Activate the dm-modules

            if( is_dir( $path ) && $dir = @opendir( $path ) ) {
                while ( ( $file = readdir( $dir ) ) !== false ) {
                    if( $file != '.' && $file != '..' && is_dir( $path . $file ) && !in_array( substr( $file, 0, 1 ), array( '_', '.' ) ) ){
                        $this->loadDirectory( $path . $file );
                    }
                }
            }
            @closedir( $path );
            do_action( 'dm_loaded', $this );
            $this->loaded = true;
        }
    }

    // Returns the current instance of DM Utils. The instance is the core
    // object which modules can register members and methods on.
    function DM( ){
        return DM_Utils::getInstance( );
    }

    DM( );  // Start DM Utils
