<?php
    /**
     *  Module Name: DM Gallery
     *  Version:     v1.0
     *  Description: Works with ACF to create bootstrap galleries.
     *  Author:      Distill Mill LLC
     *  Author URI:  http://distillmill.com
     */

    /**
     *  This addon for dm-utils tidies up the use of the ACF gallery field type.
     *  It handles the loading of a gallery based on a provided post or post ID
     *  or it will grab the current post from the loop. It can then be used to
     *  check for slides or to iterate over them.
     */
    class DM_Gallery{
        private static $galleries;  // Stores all galleries

        private $index = 0;
        private $masthead = false;
        private $postID;
        private $slides = array( );

        protected function getPostID( $post = null ){
            if( is_null( $post ) )
                return get_the_ID( );
            elseif( $post instanceof WP_Post )
                return $post->ID;

            return $post;  // It's either valid or its not let the caller figure it out.
        }


        // Use to load a gallery without creating multiple instances of the same
        // gallery in memory.
        public static function getGallery( $post = null, $masthead = false ){
            $post = self::getPostID( $post );
            $name = $post . ( $masthead ? '-masthead' : '' );
            if( is_null( self::$galleries ) )
                self::$galleries = array( );

            if( !array_key_exists( $name, self::$galleries ) )
                self::$galleries[ $name ] = new self( $post, $masthead );

            return self::$galleries[ $name ];
        }

        // Gallery constructor. Takes post as either an integer, WP_Post, or
        // null. Using null (not passing anything) will load the ID of the
        // current post in the loop.
        public function __construct( $post = null, $masthead = false ){
            $this->postID = $this->getPostID( $post );
            $this->masthead = $masthead;
        }

        // Returns the next slide object or null if there are no more.
        public function getNextSlide( ){
            $slide = null;
            if( $this->index < count( $this->getSlides( ) ) ){
                $slides = $this->getSlides( );
                $slide = $slides[ $this->index ];
                $this->index++;
            }

            return $slide;
        }

        // Gets the number of slides
        public function getNumSlides( ){
            return count( $this->getSlides( ) );
        }

        // Gets the slides from ACF if there are any and if ACF is enabled.
        public function getSlides( ){
            if( !$this->slides )
                $this->buildSlides( );

            return $this->slides;
        }

        // Determins if there are any slides. Returns true if there are.
        public function hasSlides( ){
            return count( $this->getSlides( ) ) > 0;
        }

        // Is true if the gallery is a masthead
        public function isMasthead( ){
            return $this->masthead;
        }

        // Resets the slide index to 0.
        public function resetSlides( ){
            $this->index = 0;
        }

        // Converts the ACF gallery object into useful DMGallerySlides.
        private function buildSlides( ){
            if( $this->isMasthead( ) ){
                if( DM()->getField( 'slides' ) ){
                    foreach( DM()->getField( 'slides' ) as $index => $slide ){
                        $this->slides[ ] = new DM_GallerySlide( $slide, $index, true );
                    }
                }
            }
            else{
                if( DM()->getField( 'gallery' ) ){
                    foreach( DM()->getField( 'gallery' ) as $index => $slide ){
                        $this->slides[ ] = new DM_GallerySlide( $slide, $index );
                    }
                }
            }
        }
    }

    /**
     *  Wrapper for ACF gallery slides.
     */
    class DM_GallerySlide{
        public $data;

        private $index;
        private $masthead;

        // Slide constructor. Accepts an ACF slide object and optionally its
        // index in the gallery.
        public function __construct( $slide, $index = -1, $masthead = false ){
            $this->data = $slide;
            $this->index = $index;
            $this->masthead = $masthead;
        }

        // Convenience method to get the slide's alt information.
        public function getAlt( ){
            if( $this->isMasthead( ) ){
                if( $this->getType( ) == 'custom' )
                    return $this->data[ 'title' ];
                else
                    return getFilteredTitle( $this->data[ $this->getType( ) ][ 0 ] );
            }
            return $this->data[ 'alt' ];
        }

        // Convenience method to get the slide's caption.
        public function getCaption( ){
            if( $this->isMasthead( ) ){
                if( $this->getType( ) == 'custom' )
                    return $this->data[ 'title' ];
                else
                    return getFilteredTitle( $this->data[ $this->getType( ) ][ 0 ] );
            }
            return $this->data[ 'caption' ];
        }

        // Convenience method to get the slide's description.
        public function getDescription( $length = null ){
            if( $this->isMasthead( ) ){
                if( $this->getType( ) == 'custom' )
                    return $this->data[ 'content' ];
                elseif( $length )
                    return getFilteredExcerpt( $this->data[ $this->getType( ) ][ 0 ], $length );
                else
                    return getFilteredContent( $this->data[ $this->getType( ) ][ 0 ] );
            }

            return $this->data[ 'description' ];
        }

        public function getURL( ){
            if( $this->isMasthead( ) && $this->getType( ) != 'custom' )
                return get_permalink( $this->data[ $this->getType( ) ][ 0 ] );

            return null;
        }

        // Convenience method to get the slide's image. It returns the URL and
        // optionally accepts width, height, and anchor parameters to get a
        // timthumb image URL.
        public function getImage( $width = '', $height = '', $a = '' ){
            if( $this->isMasthead( ) ){
                if( $this->getType( ) == 'custom' )
                    $image = $this->data[ 'image' ][ 'url' ];
                else
                    $image = get_post_thumbnail_url( $this->data[ $this->getType( ) ][ 0 ] );
            } else
                $image = wp_get_attachment_url( $this->data[ 'id' ] );

            $image = DM( )->sizeImage( $image, $width, $height, $a );

            return $image;
        }

        // Get the slides index in the gallery. This may be -1 if not set and
        // isn't a guarenteed index.
        public function getIndex( ){
            return $this->index;
        }

        // Convenience method to get the slide's title.
        public function getTitle( ){
            if( $this->isMasthead( ) )
                return $this->getAlt( );

            return $this->data[ 'title' ];
        }

        // Returns the slide type.
        public function getType( ){
            if( $this->isMasthead( ) )
                return $this->data[ 'slide_type' ];

            return 'slide';
        }

        // Returns true if this is a masthead slide.
        public function isMasthead( ){
            return $this->masthead;
        }
    }