<?php
    /**
     *  Module Name: DM Templates
     *  Version:     v1.0
     *  Description: A custom template system that runs on top of WordPresses default system.
     *  Author:      Distill Mill LLC
     *  Author URI:  http://distillmill.com
     */

    if( !defined( 'LOCAL' ) ) define( 'LOCAL', false );

    class DM_Template{
        private static $instance;
        private $debug = array( );

        public static function getInstance( ){
            if( !self::$instance )
                new self;

            return self::$instance;
        }

        public static function init( $dm ){
            $dm->registerMethod( 'getPageType', self::call( 'getPageType' ) );
            $dm->registerMethod( 'getPageSlug', self::call( 'getPageSlug' ) );
        }

        public static function call( $method ){
            return array( self::getInstance( ), $method );
        }

        public static function getHeader( $onlyMaster=false ){
            self::getInstance( )->getHeaderTemplate( $onlyMaster );
        }

        public static function getFooter( $onlyMaster=false ){
            self::getInstance( )->getFooterTemplate( $onlyMaster );
        }

        public static function getContent( ){
            self::getInstance( )->getContentTemplate( );
        }

        public static function getSidebar( ){
            self::getInstance( )->getSidebarTemplate( );
        }

        public static function template( $name = null ){
            self::getInstance( )->useTemplate( $name );
        }

        public function __construct( ){
            if( self::$instance )
                return;
            self::$instance = $this;

            // This is for debug purposes only...
            // Needs to ouput with the admin bar or else it'll miss some of the
            // footer locators.
            if( LOCAL )
                add_action( 'wp_before_admin_bar_render', self::call( 'doDebugOuput' ) );
        }

        public function doDebugOuput( ){
            echo '<script>';
            echo 'console.log( "Template Locator Debug" );';
            echo 'console.log( ' . json_encode( $this->debug ) . ' );';
            echo '</script>';
        }

        public function getHeaderTemplate( $onlyMaster=false ){
            // We want to load the header using get_header so that the get_header action fires
            // before anything then we use getTemplate for header-master. This retains WordPress's
            // order of doing things and avoids weird plugin behavior.
            $parts = $this->findTemplate( $this->templateLocator( 'header' ) );
            if( $parts )
                array_splice( $parts, 0, 1 );

            if( !$onlyMaster )
                get_header( ( $parts ? implode( '-', $parts ) : $parts ) );

            $this->getTemplate( 'header-master' );
        }

        public function getFooterTemplate( $onlyMaster=false ){
            // We want to load the footer-master using get_footer so that the get_footer action fires
            // before anything then we use getTemplate for footer. This retains WordPress's order
            // of doing things and avoids weird plugin behavior.
            $parts = $this->findTemplate( $this->templateLocator( 'footer-master' ) );
            if( $parts )
                array_splice( $parts, 0, 1, 'master' );
            
            if( !$onlyMaster ){
                get_footer( ( $parts ? implode( '-', $parts ) : $parts ) );
                $this->getTemplate( 'footer' );
            } else{
                $this->getTemplate( 'footer-master' );
            }
        }

        public function getContentTemplate( ){
            $this->getTemplate( 'before' );
            $this->getTemplate( 'content' );
            $this->getTemplate( 'after' );
        }

        public function getSidebarTemplate( ){
            $this->getTemplate( 'sidebar' );
        }

        public function useTemplate( $name = null ){
            $loaded = apply_filters( 'dm_load_template', false, $name );
            if( !$loaded ){
                $parts = array( 'template' );
                if( $name )
                    $parts[ ] = $name;

                $parts = $this->findTemplate( $parts );
                
                if( $parts ){
                    $this->loadTemplate( $parts );
                }
            }
        }

        public function findTemplate( $parts ){
            while( $parts ){
                if( $this->templateExists( $parts ) ){
                    break;
                }
                array_pop( $parts );
            }

            return ( $parts ? $parts : null );
        }

        public function loadTemplate( $parts ){
            $filename = implode( '-', $parts ) . '.php';
            load_template( locate_template( $filename ), false );
        }

        public function templateExists( $parts ){
            $filename = implode( '-', $parts ) . '.php';
            $path = locate_template( $filename );
            return isset( $path ) && $path != '';
        }

        public function getTemplate( $section ){
            $parts = $this->templateLocator( $section );
            $parts = $this->findTemplate( $parts );
            if( $parts )
                $this->loadTemplate( $parts );
        }

        public function getPageSlug( ){
            global $wp_query;
            if( is_archive( ) && ( is_category( ) || is_tag( ) || is_tax( ) ) ){
                return $wp_query->queried_object->slug;
            } elseif( is_single( ) || is_page( ) || ( !is_front_page( ) && is_home( ) ) ){
                return $wp_query->queried_object->post_name;
            }
        }

        public function getPageType( ){
            global $wp_query;
            if( is_post_type_archive( ) ){
                return $wp_query->query_vars[ 'post_type' ];
            } elseif( is_archive( ) ){
                return $wp_query->queried_object->taxonomy;
            } elseif( is_single( ) || is_page( ) || ( !is_front_page( ) && is_home( ) ) ){
                return $wp_query->queried_object->post_type;
            }
        }

        private function templateLocator( $section = null ){
            /**
             *  The goal is to have unique identifiers for every possible page that are
             *  constructed simply enough that it can be easily understood. These
             *  identifiers are then used to load specific template files based on what
             *  is being viewed and to allow for overriding general sections of the
             *  template (header, footer, sidebar, content) without having to duplicate
             *  large chuncks of theme code.
             *
             *  ARCHIVE
             *  {section}-archive-{type}-{slug}.php
             *  {section)-archive-{type}.php
             *  {section}-archive.php
             *  {section}.php
             *
             *  SINGLE
             *  {section}-single-{post-type}-{post-slug}.php
             *  {section}-single-{post-type}.php
             *  {section}-single.php
             *  {section}.php
             *
             *  HOME/FRONT PAGE
             *  {section}-special-home.php
             *  {section}-special.php
             *  {section}.php
             *
             *  SEARCH PAGE
             *  {section}-special-search.php
             *  {section}-special.php
             *  {section}.php
             *
             *  404 PAGE
             *  {section}-error-404.php
             *  {section}-error.php
             *  {section}.php
             *
             *  EVERYTHING ELSE
             *  {section}-{post-type}-{post-slug}.php
             *  {section}-{post-type}.php
             *  {section}.php
             *
             *  Where the sections are header, header-master, before, content, after,
             *  sidebar, footer-master, and footer.
             *
             *  ORDER OF CALLS
             *  - FILTER:  dm_apply_header( preventDefault = false, headerNames = array, isSecondary = false )
             *  - ACTION:  get_header( $headerNames = array ) - this is the wordpress action but to get the functionality we want we can't use get_header directly.
             *  - INCLUDE: header
             *  - ACTION:  dm_after_header( headerNames = array, isSecondary = false )
             *  - FILTER:  dm_apply_header( preventDefault = false, headerNames = array, isSecondary = true )
             *  - INCLUDE: header-master
             *  - ACTION:  dm_after_header( headerNames = array, isSecondary = true )
             *  - INCLUDE: before
             *  - INCLUDE: content
             *  - INCLUDE: after
             *  - INCLUDE: sidebar
             *  - FILTER:  dm_apply_footer( preventDefault = false, footerNames = array, isSecondary = true )
             *  - INCLUDE: footer-master
             *  - ACTION:  dm_after_footer( footerNames = array, isSecondary = true )
             *  - FILTER:  dm_apply_footer( preventDefault = false, footerNames = array, isSecondary = false )
             *  - ACTION:  get_footer( $footerNames = array ) - this is the wordpress action but to get the functionality we want we can't use get_footer directly.
             *  - INCLUDE: footer
             *  - ACTION:  dm_after_footer( footerNames = array, isSecondary = false )
             */

            $locator = array( );
            if( $section )
                $locator[ ] = $section;

            if( is_archive( ) ){
                $locator[ ] = 'archive';
            } elseif( is_search( ) ){
                $locator[ ] = 'search';
            } elseif( is_404( ) ){
                $locator[ ] = 'error';
                $locator[ ] = '404';
            } elseif( is_front_page( ) && is_home( ) ){
                $locator[ ] = 'home';
            } elseif( is_single( ) ){
                $locator[ ] = 'single';
            }

            if( $this->getPageType( ) )
                $locator[ ] = $this->getPageType( );
            if( $this->getPageSlug( ) )
                $locator[ ] = $this->getPageSlug( );

            if( LOCAL )
                $this->debug[ $section ] = implode( ' > ', $locator );

            return $locator;
        }
    }

    add_action( 'dm_init', 'DM_Template::getInstance' );